﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Diagnostics;
using RevisionAnalyser.Global;
using Microsoft.Win32;
using System.Threading;
using System.Drawing;

namespace RevisionAnalyser.Controls
{
    public class SolidSXControl : Panel
    {
        private delegate bool EnumWindowProc(IntPtr hWnd, IntPtr parameter);
        private delegate bool Win32Callback(IntPtr hwnd, IntPtr lParam);

        [DllImport("user32.dll", EntryPoint = "GetWindowThreadProcessId", SetLastError = true,
             CharSet = CharSet.Unicode, ExactSpelling = true,
             CallingConvention = CallingConvention.StdCall)]
        private static extern long GetWindowThreadProcessId(long hWnd, long lpdwProcessId);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern IntPtr FindWindow(string lpClassName, string lpWindowName);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern IntPtr FindWindowEx(IntPtr hwndParent, IntPtr hwndChildAfter, string lpszClass, string lpszWindow);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern IntPtr FindWindowEx(IntPtr parentHandle, IntPtr childAfter, string className, IntPtr windowTitle);

        [DllImport("user32.dll")]
        [return: MarshalAs(UnmanagedType.Bool)]
        private static extern bool EnumChildWindows(IntPtr hwndParent, EnumWindowProc lpEnumFunc, IntPtr lParam);

        [DllImport("user32.Dll")]
        [return: MarshalAs(UnmanagedType.Bool)]
        private static extern bool EnumChildWindows(IntPtr parentHandle, Win32Callback callback, IntPtr lParam);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern long SetParent(IntPtr hWndChild, IntPtr hWndNewParent);

        [DllImport("user32.dll", EntryPoint = "GetWindowLongA", SetLastError = true)]
        private static extern long GetWindowLong(IntPtr hwnd, int nIndex);

        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern int GetWindowText(IntPtr hWnd, StringBuilder lpString, int nMaxCount);

        [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Auto)]
        private static extern int GetWindowTextLength(IntPtr hWnd);

        [DllImport("user32.dll", EntryPoint = "SetWindowLongA", SetLastError = true)]
        private static extern long SetWindowLong(IntPtr hwnd, int nIndex, long dwNewLong);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern long SetWindowPos(IntPtr hwnd, long hWndInsertAfter, long x, long y, long cx, long cy, long wFlags);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern bool MoveWindow(IntPtr hwnd, int x, int y, int cx, int cy, bool repaint);

        [DllImport("user32.dll", EntryPoint = "PostMessageA", SetLastError = true)]
        private static extern bool PostMessage(IntPtr hwnd, uint Msg, long wParam, long lParam);

        [DllImport("User32.dll", EntryPoint = "SendMessage")]
        private static extern int SendMessage(int hWnd, int Msg, int wParam, ref COPYDATASTRUCT lParam);

        [DllImport("user32.dll")]
        private static extern int SendMessage(int hWnd, uint Msg, int wParam, int lParam);

        [DllImport("kernel32.dll", CharSet = CharSet.Auto)]
        private static extern int GetShortPathName([MarshalAs(UnmanagedType.LPTStr)] string path, [MarshalAs(UnmanagedType.LPTStr)] StringBuilder shortPath, int shortPathLength);

        [DllImport("user32.dll")]
        private static extern bool ShowWindowAsync(IntPtr hWnd, int nCmdShow);

        [return: MarshalAs(UnmanagedType.Bool)]
        [DllImport("user32.dll", SetLastError = true)]
        private static extern bool GetWindowInfo(IntPtr hwnd, ref WINDOWINFO pwi);

        private const int SWP_NOOWNERZORDER = 0x200;
        private const int SWP_NOREDRAW = 0x8;
        private const int SWP_NOZORDER = 0x4;
        private const int SWP_SHOWWINDOW = 0x0040;
        private const int WS_EX_MDICHILD = 0x40;
        private const int SWP_FRAMECHANGED = 0x20;
        private const int SWP_NOACTIVATE = 0x10;
        private const int SWP_ASYNCWINDOWPOS = 0x4000;
        private const int SWP_NOMOVE = 0x2;
        private const int SWP_NOSIZE = 0x1;
        private const int GWL_STYLE = (-16);
        private const int WS_VISIBLE = 0x10000000;
        private const int WM_COMMAND = 0x0112;
        private const int WM_CLOSE = 0xF060;
        private const int WS_CHILD = 0x40000000;
        private const int MF_BYPOSITION = 0x400;
        private const int MF_REMOVE = 0x1000;
        private const int WM_COPYDATA = 0x4A;
        private const int WM_SYSCOMMAND = 0x0112;
        private const int SC_CLOSE = 0xF060;

        private struct COPYDATASTRUCT
        {
            public IntPtr dwData;
            public int cbData;
            [MarshalAs(UnmanagedType.LPStr)]
            public string lpData;
        }

        [StructLayout(LayoutKind.Sequential)]
        struct WINDOWINFO
        {
            public uint cbSize;
            public RECT rcWindow;
            public RECT rcClient;
            public uint dwStyle;
            public uint dwExStyle;
            public uint dwWindowStatus;
            public uint cxWindowBorders;
            public uint cyWindowBorders;
            public ushort atomWindowType;
            public ushort wCreatorVersion;
            public WINDOWINFO(Boolean? filler)
                : this()  
            {
                cbSize = (UInt32)(Marshal.SizeOf(typeof(WINDOWINFO)));
            }
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct RECT
        {
            private int _Left;
            private int _Top;
            private int _Right;
            private int _Bottom;

            public RECT(RECT Rectangle)
                : this(Rectangle.Left, Rectangle.Top, Rectangle.Right, Rectangle.Bottom)
            {
            }
            public RECT(int Left, int Top, int Right, int Bottom)
            {
                _Left = Left;
                _Top = Top;
                _Right = Right;
                _Bottom = Bottom;
            }

            public int X
            {
                get { return _Left; }
                set { _Left = value; }
            }
            public int Y
            {
                get { return _Top; }
                set { _Top = value; }
            }
            public int Left
            {
                get { return _Left; }
                set { _Left = value; }
            }
            public int Top
            {
                get { return _Top; }
                set { _Top = value; }
            }
            public int Right
            {
                get { return _Right; }
                set { _Right = value; }
            }
            public int Bottom
            {
                get { return _Bottom; }
                set { _Bottom = value; }
            }
            public int Height
            {
                get { return _Bottom - _Top; }
                set { _Bottom = value - _Top; }
            }
            public int Width
            {
                get { return _Right - _Left; }
                set { _Right = value + _Left; }
            }
            public Point Location
            {
                get { return new Point(Left, Top); }
                set
                {
                    _Left = value.X;
                    _Top = value.Y;
                }
            }
            public Size Size
            {
                get { return new Size(Width, Height); }
                set
                {
                    _Right = value.Width + _Left;
                    _Bottom = value.Height + _Top;
                }
            }

            public static implicit operator Rectangle(RECT Rectangle)
            {
                return new Rectangle(Rectangle.Left, Rectangle.Top, Rectangle.Width, Rectangle.Height);
            }
            public static implicit operator RECT(Rectangle Rectangle)
            {
                return new RECT(Rectangle.Left, Rectangle.Top, Rectangle.Right, Rectangle.Bottom);
            }
            public static bool operator ==(RECT Rectangle1, RECT Rectangle2)
            {
                return Rectangle1.Equals(Rectangle2);
            }
            public static bool operator !=(RECT Rectangle1, RECT Rectangle2)
            {
                return !Rectangle1.Equals(Rectangle2);
            }

            public override string ToString()
            {
                return "{Left: " + _Left + "; " + "Top: " + _Top + "; Right: " + _Right + "; Bottom: " + _Bottom + "}";
            }

            public override int GetHashCode()
            {
                return ToString().GetHashCode();
            }

            public bool Equals(RECT Rectangle)
            {
                return Rectangle.Left == _Left && Rectangle.Top == _Top && Rectangle.Right == _Right && Rectangle.Bottom == _Bottom;
            }

            public override bool Equals(object Object)
            {
                if (Object is RECT)
                {
                    return Equals((RECT)Object);
                }
                else if (Object is Rectangle)
                {
                    return Equals(new RECT((Rectangle)Object));
                }

                return false;
            }
        }

        // Splitter
        private SplitContainer _splitter;

        // SolidSX process
        private Process _process;

        // Process handle
        private IntPtr _handle;

        // Radial canvas handle
        private IntPtr _canvasHandle;

        // Node navigator handle
        private IntPtr _navigatorHandle;

        // Handle capture mode
        private bool _captureMode = false;

        public SolidSXControl()
        {
            _splitter = new SplitContainer();
            _splitter.Dock = DockStyle.Fill;
            _splitter.SplitterDistance = 200;
            _splitter.Panel1MinSize = 200;
            _splitter.FixedPanel = FixedPanel.Panel1;
            _splitter.SplitterMoved += new SplitterEventHandler(_splitter_SplitterMoved);
            _splitter.Resize += new EventHandler(_splitter_Resize);
            Controls.Add(_splitter);
        }

        /// <summary>
        /// Launches the SolidSX application.
        /// This function will attempt to close a possible trial dialog and to hide SolidSX.
        /// </summary>
        public void OpenSolidSX(string file)
        {
            if (_process != null)
            {
                CloseSolidSX();
            }
            _process = Process.Start(GetProcessStartInfo());
            _process.WaitForInputIdle();
            CloseTrialDialog();
            InitSolidSX();
            OpenFile(file);
        }

        /// <summary>
        /// Closes the SolidSX process.
        /// </summary>
        public void CloseSolidSX()
        {
            if (_handle != null)
            {
                SendMessage("exit()");
                SendMessage(_handle.ToInt32(), WM_COMMAND, WM_CLOSE, 0);
                _process = null;
                _handle = IntPtr.Zero;
                _canvasHandle = IntPtr.Zero;
                _navigatorHandle = IntPtr.Zero;
            }
        }

        /// <summary>
        /// Attempts to close the trial splash dialog on start-up.
        /// </summary>
        private void CloseTrialDialog()
        {
            if (_process != null)
            {
                WINDOWINFO info = new WINDOWINFO();
                info.cbSize = (uint)Marshal.SizeOf(info);
                GetWindowInfo(_process.MainWindowHandle, ref info);
                // Check if the main window handle is a dialog of 414 x 314 pixels.
                if (info.rcWindow.Width == 414 && info.rcWindow.Height == 314)
                {
                    SendMessage(_process.MainWindowHandle.ToInt32(), WM_SYSCOMMAND, SC_CLOSE, 0);
                }
            }
        }

        /// <summary>
        /// Waits for the main window handle to become active of SolidSX.
        /// After the main window handle is known, try to hide it right away.
        /// This function will operate for a maximum of 10 seconds.
        /// </summary>
        private void InitSolidSX()
        {
            if (_process != null)
            {
                for (int i = 0; i < 100; i++)
                {
                    _process.Refresh();
                    // Check if the window title is something else than an empty string.
                    if (!_process.MainWindowTitle.Equals(string.Empty))
                    {
                        // We found the SolidSX main screen. Send a hide message.
                        _handle = _process.MainWindowHandle;
                        ShowWindowAsync(_process.MainWindowHandle, 0);
                        return;
                    }
                    Thread.Sleep(100);
                }
            }
        }

        /// <summary>
        /// Opens a database file in SolidSX.
        /// After a file is opened, this function will capture the radial view and navigator handles.
        /// </summary>
        /// <param name="file"></param>
        private void OpenFile(string file)
        {
            if (_process != null)
            {
                _captureMode = true;
                SendMessage(String.Format("ensureOpen('{0}')", file.Replace(@"\", @"\\")));
                SendMessage("showPane('toolbar','False')");
                SendMessage("showPane('tree','False')");
                SendMessage("openView('heb','Radial View')");
                SetNodeColorAttribute("<none>");
                SetNodeColormap("Rainbow");
            }
        }

        public void SelectRevision(long revision)
        {
            SetNodeColorAttribute(String.Format("revision{0}", revision));
            SendMessage(String.Format("setEdgeInclusionFilter([ (\"{0}\",{1}) ])", String.Format("revision{0}", revision), revision));
        }

        public void SetNodeColormap(string colormap)
        {
            SendMessage(String.Format("setNodeColorMap('{0}')", colormap));
        }

        public void SetNodeColorAttribute(string attribute)
        {
            SendMessage(String.Format("setNodeColorAttribute('{0}')", attribute));
        }

        public void SetEdgeInclusionFilter(string attribute, long value)
        {
            SendMessage(String.Format("setEdgeInclusionFilter([ (\"{0}\",{1}) ])", attribute, value));
        }

        public void SetNodeInclusionFilter(string attribute, long value)
        {
            SendMessage(String.Format("setNodeInclusionFilter([ (\"{0}\",{1}) ])", attribute, value));
        }

        public void ShowRevisionFilter(List<Revision> revisions)
        {
            StringBuilder filter = new StringBuilder();
            foreach (Revision revision in revisions)
            {
                filter.Append(String.Format("(\"revision{0}\",{0}), ", revision.ID));
            }
            filter.Remove(filter.ToString().LastIndexOf(","), 2);
            SendMessage(String.Format("setNodeInclusionFilter([ {0} ])", filter.ToString()));
        }

        public void ExpandAll()
        {
            if (_process != null)
            {
                SendMessage("expandAll()");
            }
        }

        public void CollapseAll()
        {
            if (_process != null)
            {
                SendMessage("collapseByNid(1,'True')");
            }
        }

        public void ExpandLevel(byte level)
        {
            if (_process != null)
            {
                SendMessage(String.Format("expandToLevel({0})", level));
            }
        }

        private string GetSolidSXPath()
        {
            try
            {
                return Registry.CurrentUser.OpenSubKey(@"Software\SolidSource\SolidSX").GetValue("").ToString() + @"\bin";
            }
            catch
            {
            }
            return string.Empty;
        }

        private ProcessStartInfo GetProcessStartInfo()
        {
            string path = GlobalVariables.Instance.GetVariable("SOLIDSX_PATH", GetSolidSXPath());
            return new ProcessStartInfo()
            {
                FileName = path + @"\SolidSX2.exe",
                WorkingDirectory = path,
                Arguments = " --listento 0 --sendto " + this.Handle.ToInt32(),
                WindowStyle = ProcessWindowStyle.Minimized
            };
        }

        private static List<IntPtr> GetChildWindows(IntPtr parent)
        {
            List<IntPtr> result = new List<IntPtr>();
            GCHandle listHandle = GCHandle.Alloc(result);
            try
            {
                EnumWindowProc childProc = new EnumWindowProc(EnumWindow);
                EnumChildWindows(parent, childProc, GCHandle.ToIntPtr(listHandle));
            }
            finally
            {
                if (listHandle.IsAllocated)
                {
                    listHandle.Free();
                }
            }
            return result;
        }

        private static bool EnumWindow(IntPtr handle, IntPtr pointer)
        {
            GCHandle gch = GCHandle.FromIntPtr(pointer);
            List<IntPtr> list = gch.Target as List<IntPtr>;
            if (list == null)
            {
                throw new InvalidCastException("GCHandle Target could not be cast as List<IntPtr>");
            }
            list.Add(handle);
            return true;
        }

        private static string GetText(IntPtr hWnd)
        {
            int length = GetWindowTextLength(hWnd);
            StringBuilder sb = new StringBuilder(length + 1);
            GetWindowText(hWnd, sb, sb.Capacity);
            return sb.ToString();
        }

        private IntPtr GetMessageHandle()
        {
            return FindWindow("wxWindowClassNR", "SOLIDSX_WMCOPYDATA");
        }

        private IntPtr GetNavigatorHandle()
        {
            if (_handle != IntPtr.Zero)
            {
                foreach (IntPtr child in GetChildWindows(_handle))
                {
                    if (GetText(child).Equals("wxVListBox"))
                    {
                        return child;
                    }
                }
            }
            return IntPtr.Zero;
        }

        private IntPtr GetCanvasHandle()
        {
            if (_handle != IntPtr.Zero)
            {
                foreach (IntPtr child in GetChildWindows(_handle))
                {
                    if (GetText(child).Equals("GLCanvas"))
                    {
                        return child;
                    }
                }
            }
            return IntPtr.Zero;
        }

        private void SendMessage(string message)
        {
            IntPtr handle = GetMessageHandle();
            if (handle != IntPtr.Zero)
            {
                SendWindowsStringMessage(handle.ToInt32(), 0, message);
            }
        }

        private int SendWindowsStringMessage(int hWnd, int wParam, string msg)
        {
            int result = 0;
            if (hWnd > 0)
            {
                byte[] sarr = System.Text.Encoding.Default.GetBytes(msg);
                int len = sarr.Length;
                COPYDATASTRUCT cds = new COPYDATASTRUCT();
                cds.dwData = (IntPtr)100;
                cds.lpData = msg;
                cds.cbData = len;
                result = SendMessage(hWnd, WM_COPYDATA, wParam, ref cds);
            }
            return result;
        }

        protected override void WndProc(ref Message m)
        {
            if (m.Msg == WM_COPYDATA)
            {
                COPYDATASTRUCT data = new COPYDATASTRUCT();
                data = (COPYDATASTRUCT)m.GetLParam(typeof(COPYDATASTRUCT));
                string dataStr = data.lpData.Substring(0, data.cbData);
                ProcessMessage(dataStr);
            }
            base.WndProc(ref m);
        }

        private void ProcessMessage(string m)
        {
            if (_captureMode && m.StartsWith("OnEdgeSelectionChanged"))
            {
                IntPtr canvasHandle = GetCanvasHandle();
                if (canvasHandle != IntPtr.Zero)
                {
                    _canvasHandle = canvasHandle;
                    SetParent(_canvasHandle, _splitter.Panel2.Handle);
                    MoveWindow(_canvasHandle, 0, 0, _splitter.Panel2.Width, _splitter.Panel2.Height, true);
                    _navigatorHandle = GetNavigatorHandle();
                    if (_navigatorHandle != IntPtr.Zero)
                    {
                        SetParent(_navigatorHandle, _splitter.Panel1.Handle);
                        MoveWindow(_navigatorHandle, 0, 0, _splitter.Panel1.Width, _splitter.Panel1.Height, true);
                    }
                    _splitter.SplitterDistance = 200;
                    _captureMode = false;
                    Loaded(this, null);
                }
            }
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            this.Invalidate();
            base.OnSizeChanged(e);
        }

        void _splitter_Resize(object sender, EventArgs e)
        {
            if (_navigatorHandle != IntPtr.Zero)
            {
                MoveWindow(_navigatorHandle, 0, 0, _splitter.Panel1.Width, _splitter.Panel1.Height, true);
            }
            if (_canvasHandle != IntPtr.Zero)
            {
                MoveWindow(_canvasHandle, 0, 0, _splitter.Panel2.Width, _splitter.Panel2.Height, true);
            }
        }

        void _splitter_SplitterMoved(object sender, SplitterEventArgs e)
        {
            _splitter.Enabled = false;
            _splitter_Resize(sender, e);
            _splitter.Enabled = true;
        }

        protected override void OnHandleDestroyed(EventArgs e)
        {
            CloseSolidSX();
            base.OnHandleDestroyed(e);
        }

        public event EventHandler Loaded;
    }
}
